// Copyright 2023-2025 The NATS Authors
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package certidp

import (
	"encoding/base64"
	"errors"
	"fmt"
	"io"
	"net/http"
	"net/url"
	"strings"
	"time"

	"golang.org/x/crypto/ocsp"
)

func FetchOCSPResponse(link *ChainLink, opts *OCSPPeerConfig, log *Log) ([]byte, error) {
	if link == nil || link.Leaf == nil || link.Issuer == nil || opts == nil || log == nil {
		return nil, errors.New(ErrInvalidChainlink)
	}

	timeout := time.Duration(opts.Timeout * float64(time.Second))
	if timeout <= 0*time.Second {
		timeout = DefaultOCSPResponderTimeout
	}

	getRequestBytes := func(u string, hc *http.Client) ([]byte, error) {
		resp, err := hc.Get(u)
		if err != nil {
			return nil, err
		}
		defer resp.Body.Close()
		if resp.StatusCode != http.StatusOK {
			return nil, fmt.Errorf(ErrBadResponderHTTPStatus, resp.StatusCode)
		}
		return io.ReadAll(resp.Body)
	}

	// Request documentation:
	// https://tools.ietf.org/html/rfc6960#appendix-A.1

	reqDER, err := ocsp.CreateRequest(link.Leaf, link.Issuer, nil)
	if err != nil {
		return nil, err
	}

	reqEnc := encodeOCSPRequest(reqDER)

	responders := *link.OCSPWebEndpoints

	if len(responders) == 0 {
		return nil, errors.New(ErrNoAvailOCSPServers)
	}

	var raw []byte
	hc := &http.Client{
		Timeout: timeout,
	}
	for _, u := range responders {
		responderURL := u.String()
		log.Debugf(DbgMakingCARequest, responderURL)
		responderURL = strings.TrimSuffix(responderURL, "/")
		raw, err = getRequestBytes(fmt.Sprintf("%s/%s", responderURL, reqEnc), hc)
		if err == nil {
			break
		}
	}
	if err != nil {
		return nil, fmt.Errorf(ErrFailedWithAllRequests, err)
	}

	return raw, nil
}

// encodeOCSPRequest encodes the OCSP request in base64 and URL-encodes it.
// This is needed to fulfill the OCSP responder's requirements for the request format. (X.690)
func encodeOCSPRequest(reqDER []byte) string {
	reqEnc := base64.StdEncoding.EncodeToString(reqDER)
	return url.QueryEscape(reqEnc)
}
